using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;
using HIPS.CommonSchemas.Hi;
using HIPS.HpiiSchemas;
using HIPS.Web.Components.Cache;
using HIPS.Web.Components.Collections;
using HIPS.Web.Data.Hips.HpiiSearch;
using HIPS.Web.Data.Hips.Reference;
using HIPS.Web.Data.WebsiteDb;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.Hi;
using HIPS.Web.ModelInterface.HpiiSearch;
using HIPS.Web.UI.Conversion.HpiiSearch;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.ViewModels.HpiiSearch;

namespace HIPS.Web.UI.Controllers
{
    public class HpiiSearchController : Controller
    {
        private IHpiiSearchService HpiiSearchService { get; set; }
        private IHiReferenceRepository HiReferenceRepository { get; set; }
        private IHospitalRepository HospitalRepository { get; set; }
        private ISettingsRepository SettingsRepository { get; set; }

        private readonly List<Setting> Settings;

        private string DefaultHospitalCodeSystem
        {
            get { return Settings.GetSettingValue(Setting.SettingCodes.DefaultHospitalCodeSystem); }
        }

        public HpiiSearchController()
            : this(new HpiiSearchService(),
                   new CachedHiReferenceRepository(new HiReferenceRepository(), new WebMemoryCacheProvider(TimeSpan.FromHours(2))),
                   new CachedHospitalRepository(new HospitalRepository(), new WebMemoryCacheProvider(TimeSpan.FromHours(2))),
                   new CachedSettingsRepository(new WebsiteDbRepository(), new WebMemoryCacheProvider(TimeSpan.FromHours(2))))
        {
        }

        public HpiiSearchController(IHpiiSearchService hpiiSearchService, IHiReferenceRepository hiReferenceRepository, IHospitalRepository hospitalRepository, ISettingsRepository settingsRepository)
        {
            HpiiSearchService = hpiiSearchService;
            HiReferenceRepository = hiReferenceRepository;
            HospitalRepository = hospitalRepository;
            SettingsRepository = settingsRepository;
            Settings = settingsRepository.GetSettings();
        }


        [HttpGet]
        public ActionResult SearchById()
        {
            // Create ViewModel
            SearchByIdViewModel viewModel = new SearchByIdViewModel();

            // Load reference data
            LoadAndAddReferenceData(viewModel);

            // Render view
            return View(viewModel);
        }

        [HttpPost]
        public ActionResult SearchById(SearchByIdViewModel searchById)
        {
            // Load reference data
            LoadAndAddReferenceData(searchById);

            // Check basic validation errors
            if (!ModelState.IsValid)
            {
                // Return view (with original ViewModel)
                return View(searchById);
            }

            // Invoke service
            HpiiQueryResponse queryResponse;
            try
            {
                queryResponse = HpiiSearchService.SearchByIdentifier(searchById.ToIdentifierQuery(), this.GetCurrentUserDetails());
            }
            catch (Exception)
            {
                // Add Model State error (TODO: Remove this when HIPS provides improved validation support)
                ModelState.AddModelError(String.Empty, "Unable to locate HPI-I.");

                // Render view (with original ViewModel)
                return View(searchById);
            }
            
            // Convert to response ViewModel
            SearchResultViewModel responseVm = new SearchResultViewModel().LoadFromResponse(queryResponse);

            // Set View Logic Properties
            responseVm.ShowAustralianAddress = false;
            responseVm.ShowInternationalAddress = false;

            // Load reference data
            LoadAndAddReferenceData(responseVm, responseVm.ShowAustralianAddress, responseVm.ShowInternationalAddress);

            // If service errors, or "Not Found", return
            if (responseVm.ResponseMessages.Errors.Any() || queryResponse.HipsResponse.ResponseCode == "WSE0035")
            {
                // Set the Response Messages on the View Model being returned
                searchById.ResponseMessages = responseVm.ResponseMessages;

                // Render view (with original ViewModel)
                return View(searchById);
            }

            // Add success to view model
            responseVm.ResponseMessages.Successes.Add("HPI-I Search result found.");

            // Render response success view
            return View("SearchResult", responseVm);
        }

        [HttpGet]
        public ActionResult SearchByDemographics()
        {
            // Create ViewModel
            SearchByDemographicsViewModel viewModel = new SearchByDemographicsViewModel();

            // Load reference data
            LoadAndAddReferenceData(viewModel);

            // Render view
            return View(viewModel);
        }

        [HttpPost]
        public ActionResult SearchByDemographics(SearchByDemographicsViewModel searchByDemographics)
        {
            // Load reference data
            LoadAndAddReferenceData(searchByDemographics);

            // Check basic validation errors
            if (!ModelState.IsValid)
            {
                // Return view (with original ViewModel)
                return View(searchByDemographics);
            }

            // Invoke service
            HpiiQueryResponse queryResponse;
            try
            {
                 queryResponse = HpiiSearchService.SearchByDemographic(searchByDemographics.ToDemographicQuery(), this.GetCurrentUserDetails());
            }
            catch (Exception)
            {
                // Add Model State error (TODO: Remove this when HIPS provides improved validation support)
                ModelState.AddModelError(String.Empty, "Unable to locate HPI-I.");

                // Render view (with original ViewModel)
                return View(searchByDemographics);
            }

            // Convert to response ViewModel
            SearchResultViewModel responseVm = new SearchResultViewModel().LoadFromResponse(queryResponse);

            // Set View Logic Properties
            responseVm.ShowInternationalAddress = responseVm.InternationalAddress.HasAnyValue();
            responseVm.ShowAustralianAddress = !responseVm.ShowInternationalAddress;

            // Load reference data
            LoadAndAddReferenceData(responseVm, responseVm.ShowAustralianAddress, responseVm.ShowInternationalAddress);

            // If service errors, or "Not Found", return
            if (responseVm.ResponseMessages.Errors.Any() || queryResponse.HipsResponse.ResponseCode == "WSE0035")
            {
                // Set the Response Messages on the View Model being returned
                searchByDemographics.ResponseMessages = responseVm.ResponseMessages;
                
                // Render view (with original ViewModel)
                return View(searchByDemographics);
            }

            // Add success to view model
            responseVm.ResponseMessages.Successes.Add("HPI-I Search result found.");

            // Render response success view
            return View("SearchResult", responseVm);
        }

        #region Helper Methods

        private SearchByIdViewModel LoadAndAddReferenceData(SearchByIdViewModel searchByIdViewModel)
        {
            // Load reference data
            List<PcehrDataStore.Schemas.Hospital> hospitals = HospitalRepository.GetHospitals(DefaultHospitalCodeSystem);
            List<HiSex> sexes = HiReferenceRepository.GetHiSexes();
            List<HiState> states = HiReferenceRepository.GetHiStates();

            // Update ViewModel with reference data
            searchByIdViewModel.Hpios = LoadHpios(hospitals);
            searchByIdViewModel.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByIdViewModel.States = states.ToSelectListItems(s => s.Code, s => s.Description);

            // Return for fluency
            return searchByIdViewModel;
        }

        private SearchByDemographicsViewModel LoadAndAddReferenceData(SearchByDemographicsViewModel searchByDemographicsViewModel)
        {
            // Load reference data
            List<PcehrDataStore.Schemas.Hospital> hospitals = HospitalRepository.GetHospitals(DefaultHospitalCodeSystem);
            List<HiSex> sexes = HiReferenceRepository.GetHiSexes();
            List<HiState> states = HiReferenceRepository.GetHiStates();
            List<HiUnitType> unitTypes = HiReferenceRepository.GetHiUnitTypes();
            List<HiLevelType> levelTypes = HiReferenceRepository.GetHiLevelTypes();
            List<HiStreetType> streetTypes = HiReferenceRepository.GetHiStreetTypes();
            List<HiStreetSuffixType> streetSuffixes = HiReferenceRepository.GetHiStreetSuffixTypes();
            List<HiPostalDeliveryType> postalDeliveryTypes = HiReferenceRepository.GetHiPostalDeliveryTypes();
            List<HiCountry> countries = HiReferenceRepository.GetHiCountries();

            // Update ViewModel with reference data
            searchByDemographicsViewModel.Hpios = LoadHpios(hospitals);
            searchByDemographicsViewModel.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.States = states.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.UnitTypes = unitTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.LevelTypes = levelTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.StreetTypes = streetTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.StreetSuffixes = streetSuffixes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.PostalDeliveryTypes = postalDeliveryTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.Countries = countries.ToSelectListItems(s => s.Code, s => s.Description);

            return searchByDemographicsViewModel;
        }

        private SearchResultViewModel LoadAndAddReferenceData(SearchResultViewModel searchResultViewModel, bool usingAustralianAddress, bool usingInternationalAddress)
        {
            // Load reference data
            List<PcehrDataStore.Schemas.Hospital> hospitals = HospitalRepository.GetHospitals(DefaultHospitalCodeSystem);
            List<HiSex> sexes = HiReferenceRepository.GetHiSexes();
            List<HiState> states = HiReferenceRepository.GetHiStates();

            // Update ViewModel with reference data
            searchResultViewModel.Hpios = LoadHpios(hospitals);
            searchResultViewModel.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            searchResultViewModel.States = states.ToSelectListItems(s => s.Code, s => s.Description);

            if (usingAustralianAddress)
            {
                // Load reference data
                List<HiUnitType> unitTypes = HiReferenceRepository.GetHiUnitTypes();
                List<HiLevelType> levelTypes = HiReferenceRepository.GetHiLevelTypes();
                List<HiStreetType> streetTypes = HiReferenceRepository.GetHiStreetTypes();
                List<HiStreetSuffixType> streetSuffixes = HiReferenceRepository.GetHiStreetSuffixTypes();
                List<HiPostalDeliveryType> postalDeliveryTypes = HiReferenceRepository.GetHiPostalDeliveryTypes();

                // Update ViewModel with reference data
                searchResultViewModel.UnitTypes = unitTypes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.LevelTypes = levelTypes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.StreetTypes = streetTypes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.StreetSuffixes = streetSuffixes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.PostalDeliveryTypes = postalDeliveryTypes.ToSelectListItems(s => s.Code, s => s.Description);
            }
            if (usingInternationalAddress)
            {
                // Load reference data
                List<HiCountry> countries = HiReferenceRepository.GetHiCountries();

                // Update ViewModel with reference data
                searchResultViewModel.Countries = countries.ToSelectListItems(s => s.Code, s => s.Description);
            }

            // Return for fluency
            return searchResultViewModel;
        }

        private static IEnumerable<SelectListItem> LoadHpios(List<PcehrDataStore.Schemas.Hospital> hospitals)
        {
            return hospitals
                .GroupBy(h => h.HpiO)
                .ToSelectListItems(s => s.Key, s => s.First().HpioName)
                .OrderBy(h => h.Value);
        }

        #endregion
        
        /// <summary>
        ///     Releases unmanaged resources and optionally releases managed resources.
        /// </summary>
        /// <param name="disposing">true to release both managed and unmanaged resources; false to release only unmanaged resources.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (HpiiSearchService != null)
                {
                    HpiiSearchService.Dispose();
                    HpiiSearchService = null;
                }

                if (HiReferenceRepository != null)
                {
                    HiReferenceRepository.Dispose();
                    HiReferenceRepository = null;
                }

                if (HospitalRepository != null)
                {
                    HospitalRepository.Dispose();
                    HospitalRepository = null;
                }

                if (SettingsRepository != null)
                {
                    SettingsRepository.Dispose();
                    SettingsRepository = null;
                }
            }

            base.Dispose(disposing);
        }
    }
}